"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const utils_1 = require("../utils");
exports.default = (0, utils_1.createRule)("prefer-unicode-codepoint-escapes", {
    meta: {
        docs: {
            description: "enforce use of unicode codepoint escapes",
            category: "Stylistic Issues",
            recommended: true,
        },
        fixable: "code",
        schema: [],
        messages: {
            disallowSurrogatePair: "Use Unicode codepoint escapes instead of Unicode escapes using surrogate pairs.",
        },
        type: "suggestion",
    },
    create(context) {
        function createVisitor(regexpContext) {
            const { node, flags, getRegexpLocation, fixReplaceNode } = regexpContext;
            if (!flags.unicode && !flags.unicodeSets) {
                return {};
            }
            return {
                onCharacterEnter(cNode) {
                    if (cNode.value >= 0x10000) {
                        if (/^(?:\\u[\dA-Fa-f]{4}){2}$/u.test(cNode.raw)) {
                            context.report({
                                node,
                                loc: getRegexpLocation(cNode),
                                messageId: "disallowSurrogatePair",
                                fix: fixReplaceNode(cNode, () => {
                                    let text = String.fromCodePoint(cNode.value)
                                        .codePointAt(0)
                                        .toString(16);
                                    if (/[A-F]/u.test(cNode.raw)) {
                                        text = text.toUpperCase();
                                    }
                                    return `\\u{${text}}`;
                                }),
                            });
                        }
                    }
                },
            };
        }
        return (0, utils_1.defineRegexpVisitor)(context, {
            createVisitor,
        });
    },
});
